package msgcrypto

import (
	"bytes"
	"encoding/binary"
	"encoding/xml"
	"fmt"
	"strings"
)

type WCMsgCrypto struct {
	Token          string
	EncodingAESKey string
	ReceiverID     string
}

type RespMsg struct {
	XMLName      xml.Name `xml:"xml"`
	Encrypt      Cdata    `xml:"Encrypt"`
	Msgsignature Cdata    `xml:"MsgSignature"`
	Nonce        Cdata    `xml:"Nonce"`
	Timestamp    string   `xml:"TimeStamp"`
}

type Cdata struct {
	Text string `xml:",cdata"`
}

func NewWCCrypto(token, encodingAESKey, receiverid string) *WCMsgCrypto {
	return &WCMsgCrypto{Token: token, EncodingAESKey: (encodingAESKey + "="), ReceiverID: receiverid}
}

func (self *WCMsgCrypto) DecryptMsg(msg_signature, timestamp, nonce, encrypt string) ([]byte, *CryptError) {

	signature := calSignature(self.Token, timestamp, nonce, encrypt)

	if strings.Compare(signature, msg_signature) != 0 {
		return nil, NewCryptError(ValidateSignatureError, "signature not equal")
	}

	plaintext, crypt_err := cbcDecrypter(encrypt, self.EncodingAESKey)
	if nil != crypt_err {
		return nil, crypt_err
	}

	_, _, msg, receiver_id, crypt_err := parsePlainText(plaintext)
	if nil != crypt_err {
		return nil, crypt_err
	}

	if len(self.ReceiverID) > 0 && strings.Compare(string(receiver_id), self.ReceiverID) != 0 {
		return nil, NewCryptError(ValidateCorpidError, "receiver_id is not equil")
	}

	return msg, nil
}

func (self *WCMsgCrypto) EncryptMsg(reply_msg, timestamp, nonce string, xmlmsg *RespMsg) *CryptError {
	rand_str := randString(16)
	var buffer bytes.Buffer
	buffer.WriteString(rand_str)

	msg_len_buf := make([]byte, 4)
	binary.BigEndian.PutUint32(msg_len_buf, uint32(len(reply_msg)))
	buffer.Write(msg_len_buf)
	buffer.WriteString(reply_msg)
	buffer.WriteString(self.ReceiverID)

	tmp_ciphertext, err := cbcEncrypter(buffer.String(), self.EncodingAESKey)
	if err != nil {
		return err
	}
	ciphertext := string(tmp_ciphertext)

	signature := calSignature(self.Token, timestamp, nonce, ciphertext)
	xmlmsg.Encrypt = Cdata{ciphertext}
	xmlmsg.Msgsignature = Cdata{signature}
	xmlmsg.Nonce = Cdata{nonce}
	xmlmsg.Timestamp = timestamp

	return nil
}

func (self *WCMsgCrypto) VerifyURL(msg_signature, timestamp, nonce, echostr string) ([]byte, *CryptError) {
	signature := calSignature(self.Token, timestamp, nonce, echostr)

	if strings.Compare(signature, msg_signature) != 0 {
		return nil, NewCryptError(ValidateSignatureError, "signature not equal")
	}

	plaintext, err := cbcDecrypter(echostr, self.EncodingAESKey)
	if nil != err {
		return nil, err
	}

	_, _, msg, receiver_id, err := parsePlainText(plaintext)
	if nil != err {
		return nil, err
	}

	if len(self.ReceiverID) > 0 && strings.Compare(string(receiver_id), self.ReceiverID) != 0 {
		fmt.Println(string(receiver_id), self.ReceiverID, len(receiver_id), len(self.ReceiverID))
		return nil, NewCryptError(ValidateCorpidError, "receiver_id is not equil")
	}

	return msg, nil
}
