package main

import (
	"context"
	"log"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/joho/godotenv"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"

	"tjai/gdapi/handlers"
)

func main() {
	if err := godotenv.Load(".env"); err != nil {
		log.Fatalf("Couldn't load .env file: %v", err)
	}
	// gin.SetMode(gin.ReleaseMode)
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	mongouri := os.Getenv("MONGODB_URI")
	client, err := mongo.Connect(ctx, options.Client().ApplyURI(mongouri))

	if err != nil {
		log.Fatalf("Mongodb initialization failed: %v", err)
	}
	defer func() {
		if err = client.Disconnect(ctx); err != nil {
			panic(err)
		}
	}()

	router := gin.Default()
	router.POST("/deviceInfo", handlers.DeviceInfoHandler(client))
	router.POST("/sensorInfoUpdate", handlers.SensorInfoUpdateHandler(client))
	router.POST("/cameraSource", handlers.CameraSourceHandler(client))
	router.POST("/realtimeDataUpload", handlers.RealtimeDataUploadHandler(client))
	router.POST("/records", handlers.RecordsHandler(client))
	router.POST("/getToken", handlers.GetToken(client))

	// test interface
	// router.POST("/records/test", handlers.RecordsTestHandler(client))

	// face api
	// router.POST("/offer", handlers.OfferHandler(client))
	// router.POST("/staff", handlers.StaffHandler(client))

	// Object detection results upload API
	router.POST("/od", handlers.ObjectDetectResultHandler(client))

	// register api
	router.POST("/register", handlers.RegisterHandler(client))

	// Query API
	router.POST("/query/rtdata", handlers.GetRTDataHandler((client)))
	router.GET("/query/devices", handlers.GetDevicesHandler((client)))
	router.POST("/query/odresult", handlers.GetOdResultHandler((client)))

	// Dingtalk & WeCom event handlers
	router.GET("/events/wecommsg", handlers.VerifyWecom)
	router.GET("/events/wecomcstm", handlers.VerifyWecom)
	// router.POST("/wecommsg", wchandlers.Verify)
	router.POST("/events/wecommsg", handlers.WCMsgHandler)
	router.POST("/events/dtevent", handlers.DTEventHandler)

	// P30 direct data post

	router.Run("0.0.0.0:3010")
}
