package handlers

import (
	"encoding/xml"
	"fmt"
	"net/http"
	"os"
	"tjai/gdapi/msgcrypto"

	"github.com/gin-gonic/gin"
)

type RespText struct {
	XMLName      xml.Name        `xml:"xml"`
	Tousername   msgcrypto.Cdata `xml:"ToUserName"`
	Fromusername msgcrypto.Cdata `xml:"FromUserName"`
	Msgtype      msgcrypto.Cdata `xml:"MsgType"`
	Content      msgcrypto.Cdata `xml:"Content"`
	Createtime   uint32          `xml:"CreateTime"`
}

type WecomMsg struct {
	XMLName      xml.Name     `xml:"xml"`
	Tousername   xml.CharData `xml:"ToUserName"`
	Agentid      xml.CharData `xml:"AgentID"`
	Encrypt      xml.CharData `xml:"Encrypt,omitempty"`
	Fromusername xml.CharData `xml:"FromUsername,omitempty"`
	Createtime   uint32       `xml:"CreateTime,omitempty"`
	Msgtype      xml.CharData `xml:"MsgType,omitempty"`
	Content      xml.CharData `xml:"Content,omitempty"`
	Msgid        uint64       `xml:"MsgId,omitempty"`
}

func getWCAuth() (string, string, string) {
	token := os.Getenv("TOKEN")
	receiverId := os.Getenv("WCCORPID")
	encodingAeskey := os.Getenv("ENCODINGAESKEY")
	return token, receiverId, encodingAeskey
}

func WCMsgHandler(ctx *gin.Context) {
	token, receiverId, encodingAeskey := getWCAuth()
	wccpt := msgcrypto.NewWCCrypto(token, encodingAeskey, receiverId)
	query := ctx.Request.URL.Query()
	// var wcmsg WecomMsg
	wcmsg := WecomMsg{}

	// print the url query 'key => value' pairs
	for key, val := range query {
		fmt.Println("key:", key, "=>", "value:", val)
	}

	msg_signatures, sigfound := query["msg_signature"]
	timestamps, tsfound := query["timestamp"]
	nonces, ncfound := query["nonce"]

	if !sigfound || !tsfound || !ncfound {
		fmt.Println("Invalid query.")
		ctx.JSON(http.StatusBadRequest, gin.H{
			"message": "Invalid query",
			"success": false,
		})
		return
	}

	if err := ctx.BindXML(&wcmsg); err != nil {
		fmt.Println("Invalid request:", err)
		ctx.JSON(http.StatusUnauthorized, gin.H{
			"message": "Unauthorized",
			"success": false,
		})
		return
	}

	msg, dcrterr := wccpt.DecryptMsg(msg_signatures[0], timestamps[0], nonces[0], string(wcmsg.Encrypt))
	if dcrterr != nil {
		fmt.Println("DecryptMsg fail", dcrterr)
		ctx.JSON(http.StatusBadRequest, gin.H{
			"message": "Decoding failed",
			"success": false,
		})
		return
	}
	fmt.Printf("Decoded XML:\n%s\n", string(msg))

	wecomxml := WecomMsg{}
	if err := xml.Unmarshal(msg, &wecomxml); err != nil {
		fmt.Println("Unmarshal xml message fail:", err)
		ctx.JSON(http.StatusBadRequest, gin.H{
			"message": "Unmarshal failed",
			"success": false,
		})
		return
	}
	fmt.Println("Received message:", string(wecomxml.Content))

	resptext := RespText{
		Tousername:   msgcrypto.Cdata{Text: string(wecomxml.Fromusername)},
		Fromusername: msgcrypto.Cdata{Text: string(wecomxml.Tousername)},
		Msgtype:      msgcrypto.Cdata{Text: "text"},
		// modify the Content key here to set resp text.
		Content:    msgcrypto.Cdata{Text: string(wecomxml.Content)},
		Createtime: wecomxml.Createtime,
	}

	respstring, resperr := xml.Marshal(&resptext)
	if resperr != nil {
		fmt.Println("Marshal xml message fail:", resperr)
		ctx.JSON(http.StatusBadRequest, gin.H{
			"message": "Marshal failed",
			"success": false,
		})
		return
	}

	xmlmsg := msgcrypto.RespMsg{}

	if err := wccpt.EncryptMsg(string(respstring), timestamps[0], nonces[0], &xmlmsg); err != nil {
		fmt.Println("Marshal xml message fail:", err)
		ctx.JSON(http.StatusBadRequest, gin.H{
			"message": "Marshal failed",
			"success": false,
		})
		return
	}

	fmt.Printf("message replying:%s\n", resptext.Content.Text)
	output, xmlerr := xml.Marshal(xmlmsg)
	if xmlerr != nil {
		fmt.Println("xmlerr:", xmlerr)
		return
	}
	fmt.Printf("Raw XML replying:\n%s\n", output)
	// os.Stdout.Write(output)
	ctx.XML(http.StatusOK, xmlmsg)
}

func VerifyWecom(ctx *gin.Context) {
	token, receiverId, encodingAeskey := getWCAuth()
	wxcpt := msgcrypto.NewWCCrypto(token, encodingAeskey, receiverId)
	query := ctx.Request.URL.Query()
	for key, val := range query {
		fmt.Println("key:", key, "=>", "value:", val)
	}

	msg_signatures, sigfound := query["msg_signature"]
	timestamps, tsfound := query["timestamp"]
	nonces, ncfound := query["nonce"]
	echostrs, esfound := query["echostr"]

	if sigfound && tsfound && ncfound && esfound {
		msg, err := wxcpt.VerifyURL(msg_signatures[0], timestamps[0], nonces[0], echostrs[0])
		if err != nil {
			fmt.Println("verifyUrl fail", err)
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthorized",
				"success": false,
			})
			return
		}
		fmt.Println("Decoded msg:", string(msg))
		ctx.String(http.StatusOK, string(msg))
	} else {
		fmt.Println("Invalid query.")
		ctx.JSON(http.StatusBadRequest, gin.H{
			"message": "Invalid query",
			"success": false,
		})
		return
	}

	// fmt.Println(query)
}
