package handlers

import (
	"context"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

func StaffHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var employee staffReq
		if err := ctx.BindJSON(&employee); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": employee,
			})
			return
		}

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, employee.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		collstaff := mongoc.Database(dbname).Collection("employees")
		opts := options.FindOneAndUpdate().SetUpsert(true).SetReturnDocument(options.Before)
		filter := bson.D{{Key: "workNo", Value: employee.Number}}

		// to be tested
		employeedoc := staffDoc{
			Device:   devid,
			Name:     employee.Name,
			Number:   employee.Number,
			User:     userid,
			Captured: employee.Captured,
		}
		update := bson.D{{Key: "$set", Value: employeedoc}}

		var staffdoc bson.M
		if err := collstaff.FindOneAndUpdate(mctx, filter, update, opts).Decode(&staffdoc); err != nil {
			if err == mongo.ErrNoDocuments {
				ctx.JSON(http.StatusOK, gin.H{
					"message": "New employee added",
					"success": true,
					"payload": employee,
				})
				return
			} else {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": err.Error(),
					"success": false,
				})
				return
			}
		}
		ctx.JSON(http.StatusOK, gin.H{
			"message": "Employee profile updated",
			"success": true,
			"payload": employee,
		})
	}
	return gin.HandlerFunc(fn)
}
