package handlers

import (
	"context"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
)

func SensorInfoUpdateHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var sensorinfo sensorInfoReq
		if err := ctx.BindJSON(&sensorinfo); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": sensorinfo,
			})
			return
		}

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, sensorinfo.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		collsensor := mongoc.Database(dbname).Collection("sensorinfos")
		// opts := options.FindOneAndUpdate().SetUpsert(false).SetReturnDocument(options.Before)
		filter := bson.D{{Key: "code", Value: sensorinfo.Code}}

		// to be tested
		sensorinfodoc := sensorInfoDoc{
			Code:            sensorinfo.Code,
			Name:            sensorinfo.Name,
			Type:            sensorinfo.Type,
			Upperthreshold:  sensorinfo.Upperthreshold,
			Bottomthreshold: sensorinfo.Bottomthreshold,
			Device:          devid,
			User:            userid,
			UpdatedAt:       primitive.NewDateTimeFromTime(time.Now()),
		}

		var sensordoc bson.M
		if err := collsensor.FindOne(mctx, filter).Decode(&sensordoc); err != nil {
			if err == mongo.ErrNoDocuments {
				sensorinfodoc.CreatedAt = primitive.NewDateTimeFromTime(time.Now())
				insres, inserr := collsensor.InsertOne(mctx, sensorinfodoc)
				if inserr != nil {
					ctx.JSON(http.StatusInternalServerError, gin.H{
						"message": "Failed to add new sensor",
						"success": false,
						"payload": inserr.Error(),
					})
					return
				}
				ctx.JSON(http.StatusOK, gin.H{
					"message": "New sensor added",
					"success": true,
					"payload": insres.InsertedID,
				})
				return
			} else {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": err.Error(),
					"success": false,
				})
				return
			}
		}

		// if snsts, snstsok := sensordoc["createdAt"].(primitive.DateTime); snstsok {
		// 	sensorinfodoc.CreatedAt = snsts
		// } else {
		// 	ctx.JSON(http.StatusInternalServerError, gin.H{
		// 		"message": "Sensordoc has invalid created timestamp.",
		// 		"success": false,
		// 		"payload": sensordoc,
		// 	})
		// 	return
		// }
		update := bson.D{{Key: "$set", Value: sensorinfodoc}}

		if snsdocid, snsdocidok := sensordoc["_id"].(primitive.ObjectID); snsdocidok {
			updtres, updterr := collsensor.UpdateByID(mctx, snsdocid, update)
			if updterr != nil {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": "Failed to update sensor.",
					"success": false,
					"payload": updterr.Error(),
				})
				return
			}
			ctx.JSON(http.StatusOK, gin.H{
				"message": "Sensor info updated",
				"success": true,
				"payload": updtres.ModifiedCount,
			})
		}
	}
	return gin.HandlerFunc(fn)
}
