package handlers

import (
	"context"
	"net/http"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/minio/minio-go/v7"
	"github.com/minio/minio-go/v7/pkg/credentials"
	"go.mongodb.org/mongo-driver/mongo"
)

func RecordsHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var recreq recordReq
		if err := ctx.BindJSON(&recreq); err != nil || recreq.Device == "" {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": recreq,
			})
			return
		}

		// Verify token validation
		_, _, autherr := devUserAuth(mongoc, tokenString, recreq.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		// if recreq.RecFile != "" && recreq.Date != "" {
		// 	ctx.JSON(http.StatusBadRequest, gin.H{
		// 		"message": "Bad request. Don't post recFile with date",
		// 		"success": false,
		// 		"payload": recreq,
		// 	})
		// 	return
		// }

		endpoint := os.Getenv("MINIO_HOST")
		accessKeyID := os.Getenv("MINIO_ACCESSKEY")
		secretAccessKey := os.Getenv("MINIO_SECRETKEY")

		mctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
		defer cancel()

		// Initialize minio client object.
		minioc, mcerr := minio.New(endpoint, &minio.Options{
			Creds:  credentials.NewStaticV4(accessKeyID, secretAccessKey, ""),
			Secure: true,
		})
		if mcerr != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Could not connect to MinIO server",
				"success": false,
				"payload": recreq,
			})
			return
		}

		// return dates of records if "date" key is empty.
		// if recreq.RecFile == "" {
		// 	dateMap := make(map[string]struct{})
		// 	for obj := range minioc.ListObjects(mctx, "records", minio.ListObjectsOptions{Prefix: recreq.Device + "/" + recreq.Date}) {
		// 		if obj.Err != nil {
		// 			ctx.JSON(http.StatusInternalServerError, gin.H{
		// 				"message": "Error occurred when requesting record list:" + obj.Err.Error(),
		// 				"success": false,
		// 				"payload": recreq,
		// 			})
		// 			return
		// 		}
		// 		d := strings.SplitN(obj.Key, "/", 2)[1]
		// 		if len(d) > 10 {
		// 			dateMap[d[:10]] = struct{}{}
		// 		}
		// 	}
		// 	recDate := make([]string, len(dateMap))
		// 	i := 0
		// 	for k := range dateMap {
		// 		recDate[i] = k
		// 		i++
		// 	}
		// 	ctx.JSON(http.StatusOK, gin.H{
		// 		"message":  "Record dates generated",
		// 		"success":  true,
		// 		"recDates": recDate,
		// 	})
		// 	return
		// }

		if recreq.RecFile == "" {
			var recList []string
			for obj := range minioc.ListObjects(mctx, "records", minio.ListObjectsOptions{Prefix: recreq.Device + "/" + recreq.Date}) {
				if obj.Err != nil {
					ctx.JSON(http.StatusInternalServerError, gin.H{
						"message": "Error occurred when requesting record list:" + obj.Err.Error(),
						"success": false,
						"payload": recreq,
					})
					return
				}
				recList = append(recList, strings.SplitN(obj.Key, "/", 2)[1])
			}
			ctx.JSON(http.StatusOK, gin.H{
				"message": "Record list generated",
				"success": true,
				"recList": recList,
			})
			return
		}

		// Set request parameters for content-disposition.
		reqParams := make(url.Values)
		// reqParams.Set("response-content-disposition", "attachment; filename=\""+recreq.RecFile+"\"")
		// reqParams.Set("response-content-disposition", "inline")

		recUrl, urlerr := minioc.PresignedGetObject(mctx, "records", recreq.Device+"/"+recreq.RecFile, time.Duration(2*60)*time.Second, reqParams)

		if urlerr != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": urlerr.Error(),
				"success": false,
				"payload": recreq,
			})
			return
		}

		ctx.JSON(http.StatusOK, gin.H{
			"message":      "Presigned URL generated",
			"success":      true,
			"presignedUrl": recUrl.String(),
		})
	}
	return gin.HandlerFunc(fn)
}
