package handlers

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"strconv"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

func RealtimeDataUploadHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var realtimedata realtimeDataReq
		if err := ctx.BindJSON(&realtimedata); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": realtimedata,
			})
			return
		}

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, realtimedata.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		collrtdata := mongoc.Database(dbname).Collection("realtimedatas")
		// opts := options.FindOneAndUpdate().SetUpsert(true).SetReturnDocument(options.Before)
		// filter := bson.D{{Key: "deviceid", Value: deviceinfo.Deviceid}}

		var rtddoc realtimeDataDoc

		// iterate over requested realtimedata.Sensordata to verify pdev number
		if n0, nerr := strconv.ParseInt(realtimedata.Sensordata[0].Sensorinfo[len(realtimedata.Sensordata[0].Sensorinfo)-2:], 10, 32); nerr == nil {
			for _, d := range realtimedata.Sensordata {
				n, err := strconv.ParseInt(d.Sensorinfo[len(d.Sensorinfo)-2:], 10, 32)
				if err != nil || n != n0 {
					ctx.JSON(http.StatusBadRequest, gin.H{
						"message": "Invalid sensorinfo ",
						"success": false,
						"payload": realtimedata,
					})
					return
				}
			}
			rtddoc.Pdev = int32(n0)
		}

		// create sensordata doc array
		rtddoc.Sensordata = make([]sensordataDoc, len(realtimedata.Sensordata))
		collsensor := mongoc.Database(dbname).Collection("sensorinfos")

		for i := range rtddoc.Sensordata {
			var sensor bson.M
			if err := collsensor.FindOne(mctx, bson.D{{Key: "code", Value: realtimedata.Sensordata[i].Sensorinfo}}).Decode(&sensor); err != nil {
				ctx.JSON(http.StatusBadRequest, gin.H{
					"message": "Sensor not found.",
					"success": false,
					"payload": err.Error(),
				})
				return
			}
			if sensorid, sok := sensor["_id"].(primitive.ObjectID); sok {
				rtddoc.Sensordata[i].Sensorinfo = sensorid
				rtddoc.Sensordata[i].Concentration = realtimedata.Sensordata[i].Concentration
				rtddoc.Sensordata[i].Unit = realtimedata.Sensordata[i].Unit

				// threshold comparing
				ut, utok := sensor["upperthreshold"].(int32)
				bt, btok := sensor["bottomthreshold"].(int32)

				// sensor name
				sn, scok := sensor["name"].(string)

				if !utok || !btok || !scok {
					ctx.JSON(http.StatusBadRequest, gin.H{
						"message": "No valid threshold.",
						"success": false,
						"payload": sensor,
					})
					return
				}

				if sn == "O2Sensor" {
					// to add genAlert()
					rtddoc.Sensordata[i].IsNormal = rtddoc.Sensordata[i].Concentration <= ut && rtddoc.Sensordata[i].Concentration >= bt
				} else {
					rtddoc.Sensordata[i].IsNormal = rtddoc.Sensordata[i].Concentration <= bt
				}
			} else {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": "Sensor id not valid.",
					"success": false,
					"payload": sensor,
				})
				return
			}
		}

		// complete the realtimedata document
		rtddoc.Device = devid
		rtddoc.User = userid
		rtddoc.CreatedAt = primitive.NewDateTimeFromTime(time.Now())
		rtddoc.UpdatedAt = primitive.NewDateTimeFromTime(time.Now())
		fmt.Printf("\nsensor found: %v\n", rtddoc)

		// set collection index
		indmodel := mongo.IndexModel{
			Keys:    bson.D{{Key: "createdAt", Value: 1}},
			Options: options.Index().SetExpireAfterSeconds(3600 * 24 * 30),
		}
		_, inderr := collrtdata.Indexes().CreateOne(mctx, indmodel)
		if inderr != nil {
			ctx.JSON(http.StatusOK, gin.H{
				"message": "Cannot create collection index for realtimedata",
				"success": false,
				"payload": inderr.Error(),
			})
			return
		}

		// add the realtimedata to the collection
		insres, inserr := collrtdata.InsertOne(mctx, rtddoc)
		if inserr != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Failed to add new realtimedata",
				"success": false,
				"payload": inserr.Error(),
			})
			return
		}
		ctx.JSON(http.StatusOK, gin.H{
			"message": "New realtimedata added",
			"success": true,
			"payload": insres.InsertedID,
		})
	}
	return gin.HandlerFunc(fn)
}
