package handlers

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
)

func GetRTDataHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var rtdatareq rtDataReq
		if err := ctx.BindJSON(&rtdatareq); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": rtdatareq,
			})
			return
		}

		if rtdatareq.End == 0 {
			rtdatareq.End = time.Now().UnixMilli()
		}

		if rtdatareq.Pdev == 0 {
			rtdatareq.Pdev = 1
		}

		fmt.Println(rtdatareq)

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, rtdatareq.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		collrtdata := mongoc.Database(dbname).Collection("realtimedatas")

		filter := bson.M{
			"device": devid,
			"user":   userid,
			"pdev":   rtdatareq.Pdev,
			"createdAt": bson.M{
				"$gte": primitive.NewDateTimeFromTime(time.UnixMilli(rtdatareq.Startfrom)),
				"$lt":  primitive.NewDateTimeFromTime(time.UnixMilli(rtdatareq.End)),
			},
		}

		// Filter by search term
		aggSearch := bson.M{"$match": filter}

		// Sort by time in (ascending if set startfrom and end, descending otherwise) order
		var aggSort bson.M
		if rtdatareq.Startfrom == 0 {
			aggSort = bson.M{"$sort": bson.M{"createdAt": -1}}
		} else {
			aggSort = bson.M{"$sort": bson.M{"createdAt": 1}}
		}

		// Result limit
		aggLimit := bson.M{"$limit": 1}

		// Populate Device and Sensorinfo field
		aggPopulateDevice := bson.M{"$lookup": bson.M{
			"from":         "devices",
			"localField":   "device",
			"foreignField": "_id",
			"as":           "device",
		}}

		aggUnwindDevice := bson.M{"$unwind": bson.M{"path": "$device"}}

		// Take first element from the populated array (there is only one)
		// aggProjectDevice := bson.M{"$project": bson.M{
		// 	"sensordata": true,
		// 	"pdev":       true,
		// 	"createdAt":  true,
		// 	"device":     bson.M{"$arrayElemAt": bson.A{"$device", 0}},
		// }}

		// aggUnwindSensor := bson.M{"$unwind": bson.M{"path": "$sensordata"}}

		aggUnwindSData := bson.M{"$unwind": bson.M{"path": "$sensordata"}}

		aggPopulateSensor := bson.M{"$lookup": bson.M{
			"from":         "sensorinfos",
			"localField":   "sensordata.sensorinfo",
			"foreignField": "_id",
			"as":           "sensordata.sensorinfo",
		}}

		aggUnwindSInfo := bson.M{"$unwind": bson.M{"path": "$sensordata.sensorinfo"}}

		aggGroup := bson.M{"$group": bson.M{
			"_id":        "$_id",
			"sensordata": bson.M{"$push": "$sensordata"},
			"pdev":       bson.M{"$first": "$pdev"},
			"createdAt":  bson.M{"$first": "$createdAt"},
			"device":     bson.M{"$first": "$device"},
		}}

		var pipeline []bson.M
		if rtdatareq.Startfrom == 0 {
			pipeline = []bson.M{
				aggSearch, aggSort, aggLimit,
				aggPopulateDevice, aggUnwindDevice,
				aggUnwindSData, aggPopulateSensor, aggUnwindSInfo, aggGroup,
			}
		} else {
			pipeline = []bson.M{
				aggSearch, aggSort,
				aggPopulateDevice, aggUnwindDevice,
				aggUnwindSData, aggPopulateSensor, aggUnwindSInfo, aggGroup,
			}
		}

		cursor, aggerr := collrtdata.Aggregate(mctx, pipeline)
		if aggerr != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Aggregation failed",
				"success": false,
				"payload": aggerr.Error(),
			})
			return
		}

		var rtdataresult []rtDataDoc

		// var rtdataresult []bson.M

		if err := cursor.All(mctx, &rtdataresult); err != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Query failed",
				"success": false,
				"payload": err.Error(),
			})
			return
		}

		ctx.IndentedJSON(http.StatusOK, gin.H{
			"message":      "Query done",
			"success":      true,
			"realtimedata": rtdataresult,
		})
	}
	return gin.HandlerFunc(fn)
}

func GetDevicesHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// Verify token validation
		username, parseerr := verifyJWT(tokenString)
		if parseerr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": parseerr.Error(),
			})
			return
		}

		// Query user doc
		dbname := os.Getenv("DB_NAME")
		colluser := mongoc.Database(dbname).Collection("users")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		var userdoc userDoc
		if err := colluser.FindOne(mctx, bson.D{{Key: "name", Value: username}}).Decode(&userdoc); err != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": err.Error(),
			})
			return
		}

		colldev := mongoc.Database(dbname).Collection("devices")

		// query devices
		var devicedocs []deviceInfoDoc
		cursor, qerr := colldev.Find(mctx, bson.M{"user": userdoc.Id})
		if qerr != nil {
			if qerr == mongo.ErrNoDocuments {
				ctx.JSON(http.StatusOK, gin.H{
					"message": "You have no device",
					"success": true,
				})
				return
			} else {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": "Query failed: \n" + qerr.Error(),
					"success": false,
				})
				return
			}
		}

		if err := cursor.All(mctx, &devicedocs); err != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Result parse failed: \n" + err.Error(),
				"success": false,
			})
			return
		}

		ctx.JSON(http.StatusOK, gin.H{
			"message": "Devices found",
			"success": true,
			"devices": devicedocs,
		})
	}
	return gin.HandlerFunc(fn)
}

func GetOdResultHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var odresreq odResQueryReq
		if err := ctx.BindJSON(&odresreq); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": odresreq,
			})
			return
		}

		if odresreq.End == 0 {
			odresreq.End = time.Now().UnixMilli()
		}

		fmt.Println(odresreq)

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, odresreq.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		collodres := mongoc.Database(dbname).Collection("odresults")

		filter := bson.M{
			"device": devid,
			"user":   userid,
			"createdAt": bson.M{
				"$gte": primitive.NewDateTimeFromTime(time.UnixMilli(odresreq.Startfrom)),
				"$lt":  primitive.NewDateTimeFromTime(time.UnixMilli(odresreq.End)),
			},
		}

		// Filter by search term
		aggSearch := bson.M{"$match": filter}

		// Sort by time in (ascending if set startfrom and end, descending otherwise) order
		var aggSort bson.M
		if odresreq.Startfrom == 0 {
			aggSort = bson.M{"$sort": bson.M{"createdAt": -1}}
		} else {
			aggSort = bson.M{"$sort": bson.M{"createdAt": 1}}
		}

		// Result limit
		aggLimit := bson.M{"$limit": 1}

		var pipeline []bson.M
		if odresreq.Startfrom == 0 {
			pipeline = []bson.M{aggSearch, aggSort, aggLimit}
		} else {
			pipeline = []bson.M{aggSearch, aggSort}
		}

		cursor, aggerr := collodres.Aggregate(mctx, pipeline)
		if aggerr != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Aggregation failed",
				"success": false,
				"payload": aggerr.Error(),
			})
			return
		}

		var odqueryres []odResultDoc

		if err := cursor.All(mctx, &odqueryres); err != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Query failed",
				"success": false,
				"payload": err.Error(),
			})
			return
		}

		ctx.JSON(http.StatusOK, gin.H{
			"message":  "Query done",
			"success":  true,
			"odresult": odqueryres,
		})
	}
	return gin.HandlerFunc(fn)
}
