package handlers

import (
	"errors"
	"fmt"
	"net"
	"net/http"
	"os/exec"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/mongo"

	"github.com/pion/interceptor"
	"github.com/pion/interceptor/pkg/intervalpli"
	"github.com/pion/rtp"
	"github.com/pion/webrtc/v4"
)

type udpConn struct {
	conn        *net.UDPConn
	port        int
	payloadType uint8
}

func genFailResp(err error) gin.H {
	respFail := gin.H{
		"message": err.Error(),
		"success": false,
	}
	return respFail
}

func OfferHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// Parse request body
		var offerreq offerReq
		if err := ctx.BindJSON(&offerreq); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": offerreq,
			})
			return
		}
		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, offerreq.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		fmt.Println("devid:", devid, "\nuserid:", userid)

		// Everything below is the Pion WebRTC API! Thanks for using it ❤️.

		// Create a MediaEngine object to configure the supported codec
		m := &webrtc.MediaEngine{}

		// Setup the codecs you want to use.
		// We"ll use a VP8 and Opus but you can also define your own
		if err := m.RegisterCodec(webrtc.RTPCodecParameters{
			RTPCodecCapability: webrtc.RTPCodecCapability{MimeType: webrtc.MimeTypeVP8, ClockRate: 90000, Channels: 0, SDPFmtpLine: "", RTCPFeedback: nil},
		}, webrtc.RTPCodecTypeVideo); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}
		if err := m.RegisterCodec(webrtc.RTPCodecParameters{
			RTPCodecCapability: webrtc.RTPCodecCapability{MimeType: webrtc.MimeTypeOpus, ClockRate: 48000, Channels: 0, SDPFmtpLine: "", RTCPFeedback: nil},
		}, webrtc.RTPCodecTypeAudio); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Create a InterceptorRegistry. This is the user configurable RTP/RTCP Pipeline.
		// This provides NACKs, RTCP Reports and other features. If you use `webrtc.NewPeerConnection`
		// this is enabled by default. If you are manually managing You MUST create a InterceptorRegistry
		// for each PeerConnection.
		i := &interceptor.Registry{}

		// Register a intervalpli factory
		// This interceptor sends a PLI every 3 seconds. A PLI causes a video keyframe to be generated by the sender.
		// This makes our video seekable and more error resilent, but at a cost of lower picture quality and higher bitrates
		// A real world application should process incoming RTCP packets from viewers and forward them to senders
		intervalPliFactory, err := intervalpli.NewReceiverInterceptor()
		if err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}
		i.Add(intervalPliFactory)

		// Use the default set of Interceptors
		if err = webrtc.RegisterDefaultInterceptors(m, i); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Create the API object with the MediaEngine
		api := webrtc.NewAPI(webrtc.WithMediaEngine(m), webrtc.WithInterceptorRegistry(i))

		// Prepare the configuration
		config := webrtc.Configuration{
			ICEServers: []webrtc.ICEServer{
				{
					URLs:       []string{"turn:tongjiai.cn:3478"},
					Username:   "tjaiturn",
					Credential: "tjaiturn",
				},
			},
		}

		// Create a new RTCPeerConnection
		peerConnection, err := api.NewPeerConnection(config)
		if err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		cmd := exec.Command("ffmpeg", "-hide_banner", "-protocol_whitelist", "file,udp,rtp", "-i", "./rtp-forwarder.sdp", "-c:a", "copy", "-b:a", "48k", "-ac", "2", "-ar", "44100", "-f", "rtsp", "rtsp://tjai:tjaiCamPub@tongjiai.cn:8554/channel/audio/"+offerreq.Device)

		defer func() {
			if pErr := cmd.Process.Kill(); pErr != nil {
				fmt.Printf("ffmpeg process end error: %v\n", pErr)
			}
			if cErr := peerConnection.Close(); cErr != nil {
				fmt.Printf("cannot close peerConnection: %v\n", cErr)
			}
		}()

		// Allow us to receive 1 audio track, and 1 video track
		if _, err = peerConnection.AddTransceiverFromKind(webrtc.RTPCodecTypeAudio); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		} else if _, err = peerConnection.AddTransceiverFromKind(webrtc.RTPCodecTypeVideo); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Create a local addr
		var laddr *net.UDPAddr
		if laddr, err = net.ResolveUDPAddr("udp", "127.0.0.1:"); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Prepare udp conns
		// Also update incoming packets with expected PayloadType, the browser may use
		// a different value. We have to modify so our stream matches what rtp-forwarder.sdp expects
		udpConns := map[string]*udpConn{
			"audio": {port: 24131, payloadType: 111},
			// "video": {port: 4002, payloadType: 96},
		}
		for _, c := range udpConns {
			// Create remote addr
			var raddr *net.UDPAddr
			if raddr, err = net.ResolveUDPAddr("udp", fmt.Sprintf("127.0.0.1:%d", c.port)); err != nil {
				ctx.JSON(http.StatusInternalServerError, genFailResp(err))
				return
			}

			// Dial udp
			if c.conn, err = net.DialUDP("udp", laddr, raddr); err != nil {
				ctx.JSON(http.StatusInternalServerError, genFailResp(err))
				return
			}
			defer func(conn net.PacketConn) {
				if closeErr := conn.Close(); closeErr != nil {
					panic(closeErr)
				}
			}(c.conn)
		}

		// Set a handler for when a new remote track starts, this handler will forward data to
		// our UDP listeners.
		// In your application this is where you would handle/process audio/video
		peerConnection.OnTrack(func(track *webrtc.TrackRemote, receiver *webrtc.RTPReceiver) {
			// Retrieve udp connection
			c, ok := udpConns[track.Kind().String()]
			if !ok {
				return
			}

			b := make([]byte, 1500)
			rtpPacket := &rtp.Packet{}
			for {
				// Read
				n, _, readErr := track.Read(b)
				if readErr != nil {
					fmt.Println(readErr)
					// ctx.JSON(http.StatusInternalServerError, genFailResp(readErr))
					return
				}

				// Unmarshal the packet and update the PayloadType
				if err = rtpPacket.Unmarshal(b[:n]); err != nil {
					fmt.Println(err)
					// ctx.JSON(http.StatusInternalServerError, genFailResp(err))
					return
				}
				rtpPacket.PayloadType = c.payloadType

				// Marshal into original buffer with updated PayloadType
				if n, err = rtpPacket.MarshalTo(b); err != nil {
					fmt.Println(err)
					// ctx.JSON(http.StatusInternalServerError, genFailResp(err))
					return
				}

				// Write
				if _, writeErr := c.conn.Write(b[:n]); writeErr != nil {
					// For this particular example, third party applications usually timeout after a short
					// amount of time during which the user doesn"t have enough time to provide the answer
					// to the browser.
					// That"s why, for this particular example, the user first needs to provide the answer
					// to the browser then open the third party application. Therefore we must not kill
					// the forward on "connection refused" errors
					var opError *net.OpError
					if errors.As(writeErr, &opError) && opError.Err.Error() == "write: connection refused" {
						continue
					}
					fmt.Println(writeErr)
					return
				}
			}
		})

		// Set the handler for ICE connection state
		// This will notify you when the peer has connected/disconnected
		peerConnection.OnICEConnectionStateChange(func(connectionState webrtc.ICEConnectionState) {
			fmt.Printf("Connection State has changed %s \n", connectionState.String())

			if connectionState == webrtc.ICEConnectionStateConnected {
				fmt.Println("Ctrl+C the remote client to stop the demo")
			}
		})

		// Set the handler for Peer connection state
		// This will notify you when the peer has connected/disconnected
		peerConnection.OnConnectionStateChange(func(s webrtc.PeerConnectionState) {
			fmt.Printf("Peer Connection State has changed: %s\n", s.String())

			if s == webrtc.PeerConnectionStateFailed {
				// Wait until PeerConnection has had no network activity for 30 seconds or another failure. It may be reconnected using an ICE Restart.
				// Use webrtc.PeerConnectionStateDisconnected if you are interested in detecting faster timeout.
				// Note that the PeerConnection may come back from PeerConnectionStateDisconnected.
				fmt.Println("Done forwarding")
				return
			}

			if s == webrtc.PeerConnectionStateClosed {
				// PeerConnection was explicitly closed. This usually happens from a DTLS CloseNotify
				fmt.Println("Done forwarding")
				return
			}
		})

		// Wait for the offer to be pasted
		offer := webrtc.SessionDescription{}
		if err := Decode(offerreq.Offer, &offer); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Set the remote SessionDescription
		if err = peerConnection.SetRemoteDescription(offer); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Create answer
		answer, err := peerConnection.CreateAnswer(nil)
		if err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Create channel that is blocked until ICE Gathering is complete
		gatherComplete := webrtc.GatheringCompletePromise(peerConnection)

		// Sets the LocalDescription, and starts our UDP listeners
		if err = peerConnection.SetLocalDescription(answer); err != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(err))
			return
		}

		// Block until ICE Gathering is complete, disabling trickle ICE
		// we do this because we only can exchange one signaling message
		// in a production application you should exchange ICE Candidates via OnICECandidate
		<-gatherComplete

		// Output the answer in base64 so we can paste it in browser
		encAnswer, encerr := Encode(*peerConnection.LocalDescription())
		if encerr != nil {
			ctx.JSON(http.StatusInternalServerError, genFailResp(encerr))
			return
		}

		// fmt.Println(encAnswer)

		ctx.JSON(http.StatusOK, gin.H{
			"message": "peerConnection answer generated",
			"success": true,
			"answer":  encAnswer,
		})

		// Start streaming
		rErr := cmd.Start()
		fmt.Printf("Command finished with error: %v", rErr)
	}
	return gin.HandlerFunc(fn)
}
