package handlers

import (
	"context"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

func ObjectDetectResultHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var odresult odResultReq
		if err := ctx.BindJSON(&odresult); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": odresult,
			})
			return
		}

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, odresult.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		// debugging return
		// ctx.JSON(http.StatusOK, gin.H{
		// 	"message": "New odresult received",
		// 	"success": true,
		// 	"payload": devid.String() + "::" + userid.String(),
		// })

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		collodres := mongoc.Database(dbname).Collection("odresults")

		// object detection result document
		odresultdoc := odResultDoc{
			Device:    devid,
			Objects:   odresult.Objects,
			User:      userid,
			CreatedAt: primitive.NewDateTimeFromTime(time.Now()),
			UpdatedAt: primitive.NewDateTimeFromTime(time.Now()),
		}

		// set collection index
		indmodel := mongo.IndexModel{
			Keys:    bson.D{{Key: "createdAt", Value: 1}},
			Options: options.Index().SetExpireAfterSeconds(3600 * 24 * 30),
		}

		_, inderr := collodres.Indexes().CreateOne(mctx, indmodel)
		if inderr != nil {
			ctx.JSON(http.StatusOK, gin.H{
				"message": "Cannot create collection index for odresults",
				"success": false,
				"payload": inderr.Error(),
			})
			return
		}

		// add the odresult to the collection
		insres, inserr := collodres.InsertOne(mctx, odresultdoc)
		if inserr != nil {
			ctx.JSON(http.StatusInternalServerError, gin.H{
				"message": "Failed to add new odresult",
				"success": false,
				"payload": inserr.Error(),
			})
			return
		}
		ctx.JSON(http.StatusOK, gin.H{
			"message": "New odresult added",
			"success": true,
			"payload": insres.InsertedID,
		})

	}
	return gin.HandlerFunc(fn)
}
