package handlers

import (
	"context"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

func DeviceInfoHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// Verify token validation
		username, parseerr := verifyJWT(tokenString)
		if parseerr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": parseerr.Error(),
			})
			return
		}

		// Parse request body
		var deviceinfo deviceInfoReq
		if err := ctx.BindJSON(&deviceinfo); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": deviceinfo,
			})
			return
		}

		// Query user doc
		dbname := os.Getenv("DB_NAME")
		colluser := mongoc.Database(dbname).Collection("users")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		var userdoc bson.M
		if err := colluser.FindOne(mctx, bson.D{{Key: "name", Value: deviceinfo.User}}).Decode(&userdoc); err != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": err.Error(),
			})
			return
		}
		userdocname, usernameok := userdoc["name"].(string)
		if !usernameok || userdocname != deviceinfo.User || userdocname != username {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthorized",
				"success": false,
			})
			return
		}

		userid, useridok := userdoc["_id"].(primitive.ObjectID)
		if !useridok || userdocname != deviceinfo.User {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": "User id not found",
				"success": false,
			})
			return
		}

		coll := mongoc.Database(dbname).Collection("devices")
		opts := options.FindOneAndUpdate().SetUpsert(true).SetReturnDocument(options.Before)
		filter := bson.D{{Key: "deviceid", Value: deviceinfo.Deviceid}}

		// to be tested
		// var deviceinfodoc deviceInfoDoc
		var update bson.D
		if deviceinfo.Coordinate.Latitude == 0 || deviceinfo.Coordinate.Longitude == 0 {
			// deviceinfodoc = deviceInfoDoc{
			// 	Deviceid:     deviceinfo.Deviceid,
			// 	Serialnumber: deviceinfo.Serialnumber,
			// 	User:         userid,
			// }
			update = bson.D{{Key: "$set", Value: bson.M{
				"deviceid":     deviceinfo.Deviceid,
				"serialnumber": deviceinfo.Serialnumber,
				"user":         userid,
			}}}
		} else {
			// deviceinfodoc = deviceInfoDoc{
			// 	Deviceid:     deviceinfo.Deviceid,
			// 	Serialnumber: deviceinfo.Serialnumber,
			// 	Coordinate:   deviceinfo.Coordinate,
			// 	User:         userid,
			// }
			update = bson.D{{Key: "$set", Value: bson.M{
				"deviceid":     deviceinfo.Deviceid,
				"serialnumber": deviceinfo.Serialnumber,
				"user":         userid,
				"coordinate":   deviceinfo.Coordinate,
				"updatedAt":    primitive.NewDateTimeFromTime(time.Now()),
			}}}
		}
		// update = bson.D{{Key: "$set", Value: deviceinfodoc}}

		var devicedoc bson.M
		if err := coll.FindOneAndUpdate(mctx, filter, update, opts).Decode(&devicedoc); err != nil {
			if err == mongo.ErrNoDocuments {
				ctx.JSON(http.StatusOK, gin.H{
					"message": "New device added",
					"success": true,
					"payload": deviceinfo,
				})
				return
			} else {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": err.Error(),
					"success": false,
				})
				return
			}
		}
		ctx.JSON(http.StatusOK, gin.H{
			"message": "Device info updated",
			"success": true,
			"payload": deviceinfo,
		})
	}
	return gin.HandlerFunc(fn)
}
