package handlers

import (
	"context"
	"net/http"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

func CameraSourceHandler(mongoc *mongo.Client) gin.HandlerFunc {
	fn := func(ctx *gin.Context) {
		// Get header token
		tokenString := ctx.GetHeader("token")

		// parse request body
		var camerasource cameraSourceReq
		if err := ctx.BindJSON(&camerasource); err != nil {
			ctx.JSON(http.StatusBadRequest, gin.H{
				"message": err.Error(),
				"success": false,
				"payload": camerasource,
			})
			return
		}

		dbname := os.Getenv("DB_NAME")
		mctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		// Verify token validation
		devid, userid, autherr := devUserAuth(mongoc, tokenString, camerasource.Device)
		if autherr != nil {
			ctx.JSON(http.StatusUnauthorized, gin.H{
				"message": "Unauthenticated",
				"success": false,
				"payload": autherr.Error(),
			})
			return
		}

		collcamera := mongoc.Database(dbname).Collection("cameras")
		opts := options.FindOneAndUpdate().SetUpsert(true).SetReturnDocument(options.Before)
		filter := bson.D{{Key: "cameraid", Value: camerasource.Cameraid}}

		// to be tested
		camerasourcedoc := cameraSourceDoc{
			Cameraid: camerasource.Cameraid,
			Url:      camerasource.Url,
			Device:   devid,
			User:     userid,
		}
		update := bson.D{{Key: "$set", Value: camerasourcedoc}}

		var cameradoc bson.M
		if err := collcamera.FindOneAndUpdate(mctx, filter, update, opts).Decode(&cameradoc); err != nil {
			if err == mongo.ErrNoDocuments {
				ctx.JSON(http.StatusOK, gin.H{
					"message": "New camera added",
					"success": true,
					"payload": camerasource,
				})
				return
			} else {
				ctx.JSON(http.StatusInternalServerError, gin.H{
					"message": err.Error(),
					"success": false,
				})
				return
			}
		}
		ctx.JSON(http.StatusOK, gin.H{
			"message": "Camera source updated",
			"success": true,
			"payload": camerasource,
		})
	}
	return gin.HandlerFunc(fn)
}
